<?php
require_once 'config.php';

if (isLoggedIn()) {
    header('Location: dashboard.php');
    exit;
}

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = sanitizeInput($_POST['username']);
    $email = sanitizeInput($_POST['email']);
    $password = $_POST['password'];
    $confirm_password = $_POST['confirm_password'];
    
    // Validation
    if (empty($username) || empty($email) || empty($password)) {
        $error = 'All fields are required';
    } elseif ($password !== $confirm_password) {
        $error = 'Passwords do not match';
    } elseif (strlen($password) < 6) {
        $error = 'Password must be at least 6 characters';
    } else {
        $users = readJSON(USERS_FILE);
        
        // Check if username exists
        foreach ($users as $user) {
            if ($user['username'] === $username) {
                $error = 'Username already exists';
                break;
            }
        }
        
        if (!$error) {
            $new_user = [
                'id' => uniqid('user_', true),
                'username' => $username,
                'email' => $email,
                'password' => password_hash($password, PASSWORD_DEFAULT),
                'created_at' => date('Y-m-d H:i:s'),
                'last_login' => null
            ];
            
            $users[] = $new_user;
            writeJSON(USERS_FILE, $users);
            logActivity($username, "New user registered");
            
            $success = 'Registration successful! You can now login.';
            $_POST = []; // Clear form
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Register - Sumal Osiant</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="style.css">
    <style>
        .register-container {
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            padding: 20px;
            background: var(--sumal-gradient);
            position: relative;
        }
        
        .register-form {
            background: rgba(255,255,255,0.95);
            backdrop-filter: blur(20px);
            border-radius: 24px;
            padding: 30px;
            margin: auto;
            width: 100%;
            max-width: 450px;
            animation: slide-up 0.6s ease;
        }
        
        .form-title {
            text-align: center;
            margin-bottom: 30px;
            color: var(--sumal-dark);
        }
        
        .password-strength {
            margin-top: 8px;
            height: 4px;
            background: var(--sumal-border);
            border-radius: 2px;
            overflow: hidden;
        }
        
        .strength-bar {
            height: 100%;
            width: 0;
            transition: width 0.3s ease;
            border-radius: 2px;
        }
        
        .strength-weak { background: var(--sumal-danger); width: 33%; }
        .strength-medium { background: var(--sumal-warning); width: 66%; }
        .strength-strong { background: var(--sumal-secondary); width: 100%; }
        
        .password-requirements {
            margin-top: 10px;
            font-size: 0.85rem;
            color: var(--sumal-gray);
        }
        
        .requirement {
            display: flex;
            align-items: center;
            gap: 8px;
            margin-bottom: 5px;
        }
        
        .requirement i {
            font-size: 0.8rem;
        }
        
        .requirement.valid {
            color: var(--sumal-secondary);
        }
        
        .requirement.invalid {
            color: var(--sumal-gray);
        }
    </style>
</head>
<body>
    <div class="register-container">
        <div class="sumal-bg-animation" id="particles"></div>
        
        <div class="login-header">
            <div class="sumal-logo">
                <i class="fas fa-search-location"></i>
                <span>Sumal Osiant</span>
            </div>
            <p style="color: white; margin-top: 10px; opacity: 0.9;">Create Your Account</p>
        </div>
        
        <div class="register-form">
            <h2 class="form-title">Join Sumal Osiant</h2>
            
            <?php if($error): ?>
            <div class="sumal-alert alert-error">
                <i class="fas fa-exclamation-circle"></i>
                <span><?php echo $error; ?></span>
            </div>
            <?php endif; ?>
            
            <?php if($success): ?>
            <div class="sumal-alert alert-success">
                <i class="fas fa-check-circle"></i>
                <span><?php echo $success; ?></span>
                <div style="margin-top: 10px;">
                    <a href="login.php" class="sumal-btn" style="padding: 10px 20px; font-size: 0.9rem;">
                        <i class="fas fa-sign-in-alt"></i>
                        Go to Login
                    </a>
                </div>
            </div>
            <?php endif; ?>
            
            <form method="POST" action="" id="registerForm">
                <div class="form-group">
                    <label class="form-label">
                        <i class="fas fa-user"></i>
                        Username
                    </label>
                    <input type="text" name="username" class="sumal-input" 
                           value="<?php echo $_POST['username'] ?? ''; ?>"
                           placeholder="Choose a username" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">
                        <i class="fas fa-envelope"></i>
                        Email Address
                    </label>
                    <input type="email" name="email" class="sumal-input" 
                           value="<?php echo $_POST['email'] ?? ''; ?>"
                           placeholder="Enter your email" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">
                        <i class="fas fa-lock"></i>
                        Password
                    </label>
                    <div class="password-wrapper">
                        <input type="password" name="password" id="password" 
                               class="sumal-input" placeholder="Create a password" required
                               oninput="checkPasswordStrength()">
                        <button type="button" class="toggle-password" onclick="togglePassword('password')">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                    <div class="password-strength">
                        <div class="strength-bar" id="strengthBar"></div>
                    </div>
                    <div class="password-requirements" id="passwordRequirements">
                        <div class="requirement invalid" id="reqLength">
                            <i class="fas fa-circle"></i>
                            <span>At least 6 characters</span>
                        </div>
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">
                        <i class="fas fa-lock"></i>
                        Confirm Password
                    </label>
                    <div class="password-wrapper">
                        <input type="password" name="confirm_password" id="confirmPassword" 
                               class="sumal-input" placeholder="Confirm your password" required>
                        <button type="button" class="toggle-password" onclick="togglePassword('confirmPassword')">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                    <div id="passwordMatch" style="margin-top: 8px; font-size: 0.85rem;"></div>
                </div>
                
                <div style="margin: 25px 0;">
                    <label style="display: flex; align-items: center; gap: 10px; color: var(--sumal-gray);">
                        <input type="checkbox" name="terms" required style="width: 18px; height: 18px;">
                        <span>I agree to the <a href="#" style="color: var(--sumal-primary);">Terms of Service</a> and <a href="#" style="color: var(--sumal-primary);">Privacy Policy</a></span>
                    </label>
                </div>
                
                <button type="submit" class="sumal-btn" style="width: 100%;" id="registerBtn">
                    <i class="fas fa-user-plus"></i>
                    Create Account
                </button>
            </form>
            
            <div class="register-link" style="margin-top: 30px;">
                Already have an account? <a href="login.php">Login here</a>
            </div>
        </div>
        
        <div class="sumal-footer">
            <div class="copyright">© <?php echo date('Y'); ?> Sumal Osiant Lookup System</div>
            <div class="developer">Developed by <span>Sumal Osiant</span></div>
        </div>
    </div>
    
    <script>
        // Initialize particles
        function initParticles() {
            const container = document.getElementById('particles');
            for(let i = 0; i < 12; i++) {
                const particle = document.createElement('div');
                particle.className = 'sumal-particle';
                particle.style.width = Math.random() * 60 + 40 + 'px';
                particle.style.height = particle.style.width;
                particle.style.left = Math.random() * 100 + '%';
                particle.style.animationDelay = Math.random() * 20 + 's';
                particle.style.opacity = Math.random() * 0.07 + 0.03;
                container.appendChild(particle);
            }
        }
        
        // Toggle password visibility
        function togglePassword(fieldId) {
            const field = document.getElementById(fieldId);
            const icon = field.parentNode.querySelector('.toggle-password i');
            if(field.type === 'password') {
                field.type = 'text';
                icon.className = 'fas fa-eye-slash';
            } else {
                field.type = 'password';
                icon.className = 'fas fa-eye';
            }
        }
        
        // Check password strength
        function checkPasswordStrength() {
            const password = document.getElementById('password').value;
            const confirm = document.getElementById('confirmPassword').value;
            const strengthBar = document.getElementById('strengthBar');
            const matchDiv = document.getElementById('passwordMatch');
            const reqLength = document.getElementById('reqLength');
            
            // Check length
            if (password.length >= 6) {
                reqLength.className = 'requirement valid';
                reqLength.innerHTML = '<i class="fas fa-check-circle"></i><span>At least 6 characters ✓</span>';
            } else {
                reqLength.className = 'requirement invalid';
                reqLength.innerHTML = '<i class="fas fa-circle"></i><span>At least 6 characters</span>';
            }
            
            // Calculate strength
            let strength = 0;
            if (password.length >= 6) strength++;
            if (password.length >= 8) strength++;
            if (/[A-Z]/.test(password)) strength++;
            if (/[0-9]/.test(password)) strength++;
            if (/[^A-Za-z0-9]/.test(password)) strength++;
            
            // Update strength bar
            strengthBar.className = 'strength-bar';
            if (strength <= 1) {
                strengthBar.classList.add('strength-weak');
            } else if (strength <= 3) {
                strengthBar.classList.add('strength-medium');
            } else {
                strengthBar.classList.add('strength-strong');
            }
            
            // Check password match
            if (confirm) {
                if (password === confirm) {
                    matchDiv.innerHTML = '<span style="color: var(--sumal-secondary);"><i class="fas fa-check-circle"></i> Passwords match</span>';
                } else {
                    matchDiv.innerHTML = '<span style="color: var(--sumal-danger);"><i class="fas fa-times-circle"></i> Passwords do not match</span>';
                }
            } else {
                matchDiv.innerHTML = '';
            }
        }
        
        // Form submission
        document.getElementById('registerForm').addEventListener('submit', function(e) {
            const btn = document.getElementById('registerBtn');
            const originalHTML = btn.innerHTML;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Creating Account...';
            btn.disabled = true;
            
            setTimeout(() => {
                btn.innerHTML = originalHTML;
                btn.disabled = false;
            }, 5000);
        });
        
        // Initialize
        document.addEventListener('DOMContentLoaded', function() {
            initParticles();
            document.querySelector('input[name="username"]').focus();
            
            // Check password match in real-time
            document.getElementById('confirmPassword').addEventListener('input', checkPasswordStrength);
        });
    </script>
</body>
</html>