<?php
/**
 * 🚗 Vehicle Info Telegram Bot
 * Developer: Sumal Dev
 * Support: @Are_lqdaa
 * Copyright © Sumal Dev - All Rights Reserved
 */

// ==================== CONFIGURATION ====================
define('BOT_TOKEN', '8536616261:AAEcGR4znWkejH9Y4JcAl77ySZ9XSAhH2I8');
define('ADMIN_ID', '8137611408'); // Your Telegram User ID for broadcast
define('BOT_USERNAME', 'Vehiclefreevip_bot');
define('DEVELOPER', 'Sumal Dev');
define('SUPPORT', '@Are_lqdaa');
define('COPYRIGHT', '© Sumal Dev | Support: @Are_lqdaa');

// Database file for users (for broadcast)
define('USERS_FILE', 'users.json');
define('BROADCAST_STATE_FILE', 'broadcast_state.json');

// ==================== TELEGRAM API ====================
function sendRequest($method, $params = []) {
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/$method";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    
    $result = curl_exec($ch);
    curl_close($ch);
    
    return json_decode($result, true);
}

function sendMessage($chat_id, $text, $keyboard = null, $parse_mode = 'HTML') {
    $params = [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => $parse_mode
    ];
    
    if ($keyboard) {
        $params['reply_markup'] = json_encode($keyboard);
    }
    
    return sendRequest('sendMessage', $params);
}

function sendChatAction($chat_id, $action = 'typing') {
    return sendRequest('sendChatAction', [
        'chat_id' => $chat_id,
        'action' => $action
    ]);
}

function editMessage($chat_id, $message_id, $text, $keyboard = null) {
    $params = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'text' => $text,
        'parse_mode' => 'HTML'
    ];
    
    if ($keyboard) {
        $params['reply_markup'] = json_encode($keyboard);
    }
    
    return sendRequest('editMessageText', $params);
}

// ==================== KEYBOARD ====================
function getMainKeyboard() {
    return [
        'keyboard' => [
            [['text' => '🔍 Search Vehicle'], ['text' => '📊 Statistics']],
            [['text' => '👨‍💻 Developer'], ['text' => '📞 Support']],
            [['text' => 'ℹ️ Help']]
        ],
        'resize_keyboard' => true,
        'one_time_keyboard' => false
    ];
}

function getAdminKeyboard() {
    return [
        'keyboard' => [
            [['text' => '🔍 Search Vehicle'], ['text' => '📊 Statistics']],
            [['text' => '👨‍💻 Developer'], ['text' => '📞 Support']],
            [['text' => '📢 Broadcast'], ['text' => '👥 Users Count']],
            [['text' => 'ℹ️ Help']]
        ],
        'resize_keyboard' => true,
        'one_time_keyboard' => false
    ];
}

function getCancelKeyboard() {
    return [
        'keyboard' => [
            [['text' => '❌ Cancel']]
        ],
        'resize_keyboard' => true
    ];
}

// ==================== USER MANAGEMENT ====================
function saveUser($user_id, $username = '', $first_name = '') {
    $users = getUsers();
    
    if (!isset($users[$user_id])) {
        $users[$user_id] = [
            'id' => $user_id,
            'username' => $username,
            'first_name' => $first_name,
            'joined' => date('Y-m-d H:i:s'),
            'searches' => 0
        ];
        file_put_contents(USERS_FILE, json_encode($users, JSON_PRETTY_PRINT));
    }
    
    return $users;
}

function getUsers() {
    if (file_exists(USERS_FILE)) {
        return json_decode(file_get_contents(USERS_FILE), true) ?? [];
    }
    return [];
}

function incrementSearchCount($user_id) {
    $users = getUsers();
    if (isset($users[$user_id])) {
        $users[$user_id]['searches']++;
        file_put_contents(USERS_FILE, json_encode($users, JSON_PRETTY_PRINT));
    }
}

function setBroadcastState($user_id, $state) {
    $states = [];
    if (file_exists(BROADCAST_STATE_FILE)) {
        $states = json_decode(file_get_contents(BROADCAST_STATE_FILE), true) ?? [];
    }
    $states[$user_id] = $state;
    file_put_contents(BROADCAST_STATE_FILE, json_encode($states));
}

function getBroadcastState($user_id) {
    if (file_exists(BROADCAST_STATE_FILE)) {
        $states = json_decode(file_get_contents(BROADCAST_STATE_FILE), true) ?? [];
        return $states[$user_id] ?? null;
    }
    return null;
}

function clearBroadcastState($user_id) {
    if (file_exists(BROADCAST_STATE_FILE)) {
        $states = json_decode(file_get_contents(BROADCAST_STATE_FILE), true) ?? [];
        unset($states[$user_id]);
        file_put_contents(BROADCAST_STATE_FILE, json_encode($states));
    }
}

// ==================== VEHICLE INFO FETCHER ====================
function getVehicleDetails($rc_number) {
    $rc = strtoupper(trim($rc_number));
    $url = "https://vahanx.in/rc-search/" . $rc;
    
    $headers = [
        "Host: vahanx.in",
        "Connection: keep-alive",
        "sec-ch-ua: \"Chromium\";v=\"130\", \"Google Chrome\";v=\"130\", \"Not?A_Brand\";v=\"99\"",
        "sec-ch-ua-mobile: ?1",
        "sec-ch-ua-platform: \"Android\"",
        "Upgrade-Insecure-Requests: 1",
        "User-Agent: Mozilla/5.0 (Linux; Android 10; K) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/130.0.0.0 Mobile Safari/537.36",
        "Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,*/*;q=0.8",
        "Referer: https://vahanx.in/rc-search",
        "Accept-Encoding: gzip, deflate, br",
        "Accept-Language: en-US,en;q=0.9"
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_ENCODING, 'gzip, deflate');
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        return ['error' => "Network error: $error"];
    }
    
    if ($httpCode !== 200) {
        return ['error' => "HTTP Error: $httpCode"];
    }
    
    return parseVehicleData($response);
}

function parseVehicleData($html) {
    $data = [];
    
    $fields = [
        "Owner Name", "Father's Name", "Owner Serial No", "Model Name", "Maker Model",
        "Vehicle Class", "Fuel Type", "Fuel Norms", "Registration Date", "Insurance Company",
        "Insurance No", "Insurance Expiry", "Insurance Upto", "Fitness Upto", "Tax Upto",
        "PUC No", "PUC Upto", "Financier Name", "Registered RTO", "Address", "City Name", "Phone"
    ];
    
    foreach ($fields as $field) {
        $value = extractValue($html, $field);
        if ($value) {
            $data[$field] = $value;
        }
    }
    
    if (empty($data)) {
        return ['error' => 'No vehicle data found or invalid RC number'];
    }
    
    return $data;
}

function extractValue($html, $label) {
    // Pattern to find the label and its associated value
    $pattern = '/<span[^>]*>\s*' . preg_quote($label, '/') . '\s*<\/span>.*?<p[^>]*>([^<]+)<\/p>/is';
    
    if (preg_match($pattern, $html, $matches)) {
        return trim($matches[1]);
    }
    
    // Alternative pattern
    $pattern2 = '/' . preg_quote($label, '/') . '.*?<p[^>]*class="[^"]*"[^>]*>([^<]+)<\/p>/is';
    
    if (preg_match($pattern2, $html, $matches)) {
        return trim($matches[1]);
    }
    
    return null;
}

function formatVehicleData($data, $rc_number) {
    if (isset($data['error'])) {
        return "❌ <b>Error:</b> " . $data['error'] . "\n\n" . COPYRIGHT;
    }
    
    $rc = strtoupper(trim($rc_number));
    
    $message = "🚗 <b>Vehicle Information</b>\n";
    $message .= "━━━━━━━━━━━━━━━━━━━━\n";
    $message .= "🔢 <b>RC Number:</b> <code>$rc</code>\n\n";
    
    $icons = [
        "Owner Name" => "👤",
        "Father's Name" => "👨",
        "Owner Serial No" => "🔢",
        "Model Name" => "🚘",
        "Maker Model" => "�icing",
        "Vehicle Class" => "📋",
        "Fuel Type" => "⛽",
        "Fuel Norms" => "🌿",
        "Registration Date" => "📅",
        "Insurance Company" => "🏢",
        "Insurance No" => "📄",
        "Insurance Expiry" => "⏰",
        "Insurance Upto" => "📆",
        "Fitness Upto" => "✅",
        "Tax Upto" => "💰",
        "PUC No" => "🌱",
        "PUC Upto" => "📅",
        "Financier Name" => "🏦",
        "Registered RTO" => "🏛️",
        "Address" => "📍",
        "City Name" => "🌆",
        "Phone" => "📱"
    ];
    
    foreach ($data as $key => $value) {
        if ($value && $value !== 'N/A' && $value !== '-') {
            $icon = $icons[$key] ?? "▫️";
            $message .= "$icon <b>$key:</b> $value\n";
        }
    }
    
    $message .= "\n━━━━━━━━━━━━━━━━━━━━\n";
    $message .= "👨‍💻 <b>Developer:</b> " . DEVELOPER . "\n";
    $message .= "📞 <b>Support:</b> " . SUPPORT . "\n";
    $message .= "━━━━━━━━━━━━━━━━━━━━\n";
    $message .= COPYRIGHT;
    
    return $message;
}

// ==================== COMMAND HANDLERS ====================
function handleStart($chat_id, $user) {
    saveUser($chat_id, $user['username'] ?? '', $user['first_name'] ?? '');
    
    $keyboard = ($chat_id == ADMIN_ID) ? getAdminKeyboard() : getMainKeyboard();
    
    $message = "🚗 <b>Welcome to Vehicle Info Bot!</b>\n\n";
    $message .= "I can help you find vehicle information using RC number.\n\n";
    $message .= "🔍 <b>How to use:</b>\n";
    $message .= "• Click '🔍 Search Vehicle' button\n";
    $message .= "• Or directly send RC number (e.g., MH12AB1234)\n\n";
    $message .= "━━━━━━━━━━━━━━━━━━━━\n";
    $message .= "👨‍💻 <b>Developer:</b> " . DEVELOPER . "\n";
    $message .= "📞 <b>Support:</b> " . SUPPORT . "\n";
    $message .= "━━━━━━━━━━━━━━━━━━━━\n";
    $message .= COPYRIGHT;
    
    sendMessage($chat_id, $message, $keyboard);
}

function handleHelp($chat_id) {
    $message = "ℹ️ <b>Help & Commands</b>\n\n";
    $message .= "📝 <b>Available Commands:</b>\n";
    $message .= "/start - Start the bot\n";
    $message .= "/help - Show this help\n";
    $message .= "/search - Search vehicle info\n";
    $message .= "/dev - Developer info\n\n";
    $message .= "🔍 <b>How to Search:</b>\n";
    $message .= "Simply send the RC number directly!\n";
    $message .= "Example: <code>MH12AB1234</code>\n\n";
    $message .= "━━━━━━━━━━━━━━━━━━━━\n";
    $message .= COPYRIGHT;
    
    sendMessage($chat_id, $message);
}

function handleDeveloper($chat_id) {
    $message = "👨‍💻 <b>Developer Information</b>\n\n";
    $message .= "━━━━━━━━━━━━━━━━━━━━\n";
    $message .= "👤 <b>Name:</b> " . DEVELOPER . "\n";
    $message .= "📞 <b>Support:</b> " . SUPPORT . "\n";
    $message .= "━━━━━━━━━━━━━━━━━━━━\n\n";
    $message .= "🌟 Thanks for using this bot!\n\n";
    $message .= COPYRIGHT;
    
    sendMessage($chat_id, $message);
}

function handleSupport($chat_id) {
    $message = "📞 <b>Support & Contact</b>\n\n";
    $message .= "━━━━━━━━━━━━━━━━━━━━\n";
    $message .= "For any queries, issues or suggestions:\n\n";
    $message .= "👉 Contact: " . SUPPORT . "\n";
    $message .= "━━━━━━━━━━━━━━━━━━━━\n\n";
    $message .= COPYRIGHT;
    
    sendMessage($chat_id, $message);
}

function handleStatistics($chat_id) {
    $users = getUsers();
    $totalUsers = count($users);
    $totalSearches = 0;
    
    foreach ($users as $user) {
        $totalSearches += $user['searches'] ?? 0;
    }
    
    $message = "📊 <b>Bot Statistics</b>\n\n";
    $message .= "━━━━━━━━━━━━━━━━━━━━\n";
    $message .= "👥 <b>Total Users:</b> $totalUsers\n";
    $message .= "🔍 <b>Total Searches:</b> $totalSearches\n";
    $message .= "━━━━━━━━━━━━━━━━━━━━\n\n";
    $message .= COPYRIGHT;
    
    sendMessage($chat_id, $message);
}

function handleUsersCount($chat_id) {
    if ($chat_id != ADMIN_ID) {
        sendMessage($chat_id, "❌ You are not authorized!");
        return;
    }
    
    $users = getUsers();
    $totalUsers = count($users);
    
    $message = "👥 <b>Users Statistics</b>\n\n";
    $message .= "━━━━━━━━━━━━━━━━━━━━\n";
    $message .= "📊 <b>Total Users:</b> $totalUsers\n";
    $message .= "━━━━━━━━━━━━━━━━━━━━\n\n";
    $message .= COPYRIGHT;
    
    sendMessage($chat_id, $message);
}

function handleSearchPrompt($chat_id) {
    $message = "🔍 <b>Vehicle Search</b>\n\n";
    $message .= "Please send the RC/Registration number:\n\n";
    $message .= "📝 <b>Format Examples:</b>\n";
    $message .= "• <code>MH12AB1234</code>\n";
    $message .= "• <code>DL5CAB1234</code>\n";
    $message .= "• <code>KA01MG5678</code>\n\n";
    $message .= COPYRIGHT;
    
    sendMessage($chat_id, $message);
}

function handleBroadcast($chat_id) {
    if ($chat_id != ADMIN_ID) {
        sendMessage($chat_id, "❌ You are not authorized to use this command!");
        return;
    }
    
    setBroadcastState($chat_id, 'waiting_message');
    
    $message = "📢 <b>Broadcast Message</b>\n\n";
    $message .= "Send the message you want to broadcast to all users.\n\n";
    $message .= "You can send:\n";
    $message .= "• Text messages\n";
    $message .= "• Photos with caption\n\n";
    $message .= "Click ❌ Cancel to abort.";
    
    sendMessage($chat_id, $message, getCancelKeyboard());
}

function processBroadcast($chat_id, $message_text) {
    if ($chat_id != ADMIN_ID) {
        return;
    }
    
    clearBroadcastState($chat_id);
    
    $users = getUsers();
    $success = 0;
    $failed = 0;
    
    $keyboard = ($chat_id == ADMIN_ID) ? getAdminKeyboard() : getMainKeyboard();
    
    sendMessage($chat_id, "📤 <b>Broadcasting message to " . count($users) . " users...</b>", $keyboard);
    
    foreach ($users as $user_id => $user) {
        $broadcastMessage = $message_text . "\n\n━━━━━━━━━━━━━━━━━━━━\n" . COPYRIGHT;
        $result = sendMessage($user_id, $broadcastMessage);
        
        if ($result['ok'] ?? false) {
            $success++;
        } else {
            $failed++;
        }
        
        // Delay to avoid rate limiting
        usleep(50000); // 50ms delay
    }
    
    $report = "📢 <b>Broadcast Complete!</b>\n\n";
    $report .= "━━━━━━━━━━━━━━━━━━━━\n";
    $report .= "✅ <b>Successful:</b> $success\n";
    $report .= "❌ <b>Failed:</b> $failed\n";
    $report .= "📊 <b>Total:</b> " . count($users) . "\n";
    $report .= "━━━━━━━━━━━━━━━━━━━━\n\n";
    $report .= COPYRIGHT;
    
    sendMessage($chat_id, $report);
}

function handleVehicleSearch($chat_id, $rc_number) {
    sendChatAction($chat_id, 'typing');
    
    // Validate RC number format (basic validation)
    if (strlen($rc_number) < 6 || strlen($rc_number) > 15) {
        $message = "❌ <b>Invalid RC Number Format</b>\n\n";
        $message .= "Please enter a valid RC number.\n\n";
        $message .= "📝 <b>Format Examples:</b>\n";
        $message .= "• <code>MH12AB1234</code>\n";
        $message .= "• <code>DL5CAB1234</code>\n\n";
        $message .= COPYRIGHT;
        
        sendMessage($chat_id, $message);
        return;
    }
    
    // Remove spaces and special characters
    $rc_number = preg_replace('/[^A-Za-z0-9]/', '', $rc_number);
    
    incrementSearchCount($chat_id);
    
    $searchMsg = "🔍 <b>Searching...</b>\n\n";
    $searchMsg .= "🚗 RC Number: <code>" . strtoupper($rc_number) . "</code>\n\n";
    $searchMsg .= "⏳ Please wait...";
    
    sendMessage($chat_id, $searchMsg);
    
    $vehicleData = getVehicleDetails($rc_number);
    $response = formatVehicleData($vehicleData, $rc_number);
    
    sendMessage($chat_id, $response);
}

// ==================== MAIN WEBHOOK HANDLER ====================
$update = json_decode(file_get_contents('php://input'), true);

if (!$update) {
    echo "No update received";
    exit;
}

// Handle message updates
if (isset($update['message'])) {
    $message = $update['message'];
    $chat_id = $message['chat']['id'];
    $user = $message['from'];
    $text = $message['text'] ?? '';
    
    // Save user
    saveUser($chat_id, $user['username'] ?? '', $user['first_name'] ?? '');
    
    // Check broadcast state
    $broadcastState = getBroadcastState($chat_id);
    
    if ($broadcastState === 'waiting_message' && $chat_id == ADMIN_ID) {
        if ($text === '❌ Cancel') {
            clearBroadcastState($chat_id);
            $keyboard = getAdminKeyboard();
            sendMessage($chat_id, "❌ Broadcast cancelled.", $keyboard);
        } else {
            processBroadcast($chat_id, $text);
        }
        exit;
    }
    
    // Handle commands and buttons
    switch ($text) {
        case '/start':
            handleStart($chat_id, $user);
            break;
            
        case '/help':
        case 'ℹ️ Help':
            handleHelp($chat_id);
            break;
            
        case '/dev':
        case '👨‍💻 Developer':
            handleDeveloper($chat_id);
            break;
            
        case '📞 Support':
            handleSupport($chat_id);
            break;
            
        case '📊 Statistics':
            handleStatistics($chat_id);
            break;
            
        case '👥 Users Count':
            handleUsersCount($chat_id);
            break;
            
        case '/search':
        case '🔍 Search Vehicle':
            handleSearchPrompt($chat_id);
            break;
            
        case '/broadcast':
        case '📢 Broadcast':
            handleBroadcast($chat_id);
            break;
            
        case '❌ Cancel':
            $keyboard = ($chat_id == ADMIN_ID) ? getAdminKeyboard() : getMainKeyboard();
            sendMessage($chat_id, "❌ Cancelled.", $keyboard);
            break;
            
        default:
            // Check if it looks like an RC number
            if (preg_match('/^[A-Za-z]{2}[0-9]{1,2}[A-Za-z]{0,3}[0-9]{1,4}$/i', preg_replace('/\s+/', '', $text))) {
                handleVehicleSearch($chat_id, $text);
            } else if (strlen($text) >= 6 && strlen($text) <= 15 && preg_match('/[A-Za-z]/', $text) && preg_match('/[0-9]/', $text)) {
                // Might be an RC number with different format
                handleVehicleSearch($chat_id, $text);
            } else {
                $message = "🤔 <b>I didn't understand that!</b>\n\n";
                $message .= "Please send a valid RC number or use the buttons below.\n\n";
                $message .= "📝 <b>Example:</b> <code>MH12AB1234</code>\n\n";
                $message .= COPYRIGHT;
                
                $keyboard = ($chat_id == ADMIN_ID) ? getAdminKeyboard() : getMainKeyboard();
                sendMessage($chat_id, $message, $keyboard);
            }
            break;
    }
}

?>